<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Models;

use Exception;
use RedBeanPHP\OODBBean;
use Erlage\Photogram\System;
use Erlage\Photogram\Data\Query;

abstract class AbstractFinder
{
    /**
     * abstract finder holds reference to bean which can be passed to 
     * concrete implementation for model creation
     */
    /**
     * @property OODBBean[] 
     * */
    protected $beans;

    /**
     * @var string
     */
    protected $count;

    /**
     * @var array
     */
    protected $deSerializers;

    /**
     * @var string
     */
    protected $queryAddonLimit;

    /**
     * @var string
     */
    protected $queryAddonOrderType;

    /**
     * @var string
     */
    protected $queryAddonOrderByField;

    /**
     * returns table name 
     */
    abstract public function getTableName(): string;

    /**
     * tells whether found any models from previous find call
     */
    public function isFound(): bool
    {
        return \count($this -> beans) > 0;
    }

    /**
     * @see self::isFound()
     */
    public function notFound(): bool
    {
        return ! $this -> isFound();
    }

    /**
     * tells whether found exactly one model from previous find call
     */
    public function isFoundOne(): bool
    {
        return \count($this -> beans) == 1;
    }

    /**
     * clean finder state
     */
    public function clean(): void
    {
        unset(
            $this -> beans,
            $this -> count,
            $this -> queryAddonLimit,
            $this -> queryAddonOrderType,
            $this -> queryAddonOrderByField
        );
    }

    /**
     * @return static 
     */
    public function limit(string $limit)
    {
        $this -> queryAddonLimit = $limit;

        return $this;
    }

    /**
     * @return static 
     */
    public function orderByAsc(string $orderBy)
    {
        $this -> queryAddonOrderType = 'ASC';

        $this -> queryAddonOrderByField = $orderBy;

        return $this;
    }

    /**
     * @return static 
     */
    public function orderByDesc(string $orderBy)
    {
        $this -> queryAddonOrderType = 'DESC';

        $this -> queryAddonOrderByField = $orderBy;

        return $this;
    }

    /**
     * tells whether there are models that can be popped
     */
    public function canPop(): bool
    {
        return \count($this -> beans) > 0;
    }

    /**
     * pops model from results
     */
    abstract public function popModelFromResults();

    /**
     * returns attribute value
     * 
     * @return mixed 
     */
    abstract public function getAtrributeValue(string $attribute);

    /**
     * finds model using data provided to finder instance.
     * status of results can be checked using isFound() call and
     * results can be fetched using getModel call
     * 
     * @return static 
     */
    public function find()
    {
        $this -> process();

        return $this;
    }

    /**
     * returns number of rows found
     */
    public function findCount(): string
    {
        $this -> process(true);

        return $this -> count;
    }

    /**
     * tells whether there are models that can be popped
     */
    protected function pop(): OODBBean
    {
        return \array_pop($this -> beans);
    }

    /**
     * internal processor
     * 
     * @return static 
     */
    private function process($countOnly = false)
    {
        if (isset($this -> beans))
        {
            throw new Exception('Please clean finder state using clean() method');
        }

        if ( ! isset($this -> deSerializers))
        {
            $this -> init();
        }

        /**
         * create query object
         */
        $query = (new Query()) -> from($this -> getTableName());

        /**
         * get table class
         */
        $tableClass = System::getTableClassFromTableName($this -> getTableName());

        /**
         * first check whether primary attribute value is set in finder
         */
        $primaryAttributeValue = $this -> getAtrributeValue($tableClass::getPrimaryAttribute());

        /**
         * if primary attribute value is available then search using that
         */
        if (null != $primaryAttributeValue)
        {
            $query -> where($tableClass::getPrimaryAttribute(), $primaryAttributeValue);
        }
        /**
         * else check for other attributes
         */
        else
        {
            /**
             * add predicates for each attribute whos values is set in finder
             */
            foreach ($tableClass::allAttributes() as $databaseAttribute => $attributeAsCamelCase)
            {
                $attributeValue = $this -> getAtrributeValue($databaseAttribute);

                if (null != $attributeValue)
                {
                    /**
                     * do deSerialization of value if possible
                     */
                    if (isset($this -> deSerializers[$databaseAttribute]))
                    {
                        $deSerializedValue = $this -> deSerializers[$databaseAttribute]($attributeValue);
                    }
                    else
                    {
                        $deSerializedValue = (string) $attributeValue;
                    }

                    /**
                     * use deSerialized version in query
                     */
                    $query -> where($databaseAttribute, $deSerializedValue);
                }
            }
        }

        if (isset($this -> queryAddonLimit))
        {
            $query -> limit($this -> queryAddonLimit);
        }

        if (isset($this -> queryAddonOrderByField))
        {
            if ('ASC' == $this -> queryAddonOrderType)
            {
                $query -> orderByAsc($this -> queryAddonOrderByField);
            }
            else
            {
                $query -> orderByDesc($this -> queryAddonOrderByField);
            }
        }

        if ($countOnly)
        {
            $this -> count = $query -> count();
        }
        else
        {
            $this -> beans = $query -> select();
        }

        return $this;
    }

    private function init()
    {
        $this -> deSerializers = System::getTableClassFromTableName($this -> getTableName())::deSerializers();
    }
}
